##########################################################################################
#
#  PROGRAM:  This program implements the fast two step model selection scheme 
#            for beta regression model with varying dispersion 
#            introduced in Bayer and Cribari-Neto (2014a) (http://arxiv.org/abs/1405.3718)
#            and bootstrap-based model selection criteria 
#            introduced in Bayer and Cribari-Neto (2014b) (DOI: ???).
#            The possible choices for model selection criteria in steps 1 and 2 are:
#            "R2_FC_aj"
#            "R2_McF_aj"
#            "R2_LR_aj"
#            "R2_HS0"
#            "R2_HS1"
#            "R2_HS2"
#            "AIC"
#            "AICc"
#            "SIC"
#            "SICc"
#            "HQ"
#            "HQc"
#            "R2_D"
#            "R2_LRw" 
#            "BQCV": bootstrap-based criterion in Bayer and Cribari-Neto (2014b)
#            "632QCV": bootstrap-based criterion in Bayer and Cribari-Neto (2014b)
#
#            For details about two-step scheme, see Bayer and Cribari-Neto (2014a).
#            For details about bootstrap-based criteria, see Bayer and Cribari-Neto (2014b).
#
#            References:
#            
#            Bayer, F.M. and Cribari-Neto, F. (2014a) Model selection criteria 
#            in beta regression with varying dispersion. http://arxiv.org/abs/1405.3718
#
#            Bayer, F.M. and Cribari-Neto, F. (2014b) Bootstrap-based model 
#            selection criteria for beta regressions. http://arxiv.org/abs/1405.4525
#  
#  PS:       The model must be fitted using the GAMLSS function (in GAMLSS package)
#
#  AUTHOR:   Fabio Mariano Bayer
#
#  E-MAIL:   bayer@ufsm.br
# 
#  DATE:     May 2014
#
#########################################################################################

# ##########################################################################################
# # Usage examples:
# 
# ############################################
# ## 1) dyslexia data
# ## Application presented in Bayer and Cribari-Neto (2014a)
# data<-read.table("dyslex.txt",h=T) 
# #attach(data)
# 
# library(gamlss)
# # fitting the full model
# fit_dyslex_full <- gamlss(resp~QI+dyslex+intera+QI2+invQI+interaQI2, family=BE)
# 
# # using two step scheme (PS5) sugested in Bayer and Cribari-Neto (2014a)
# auto.beta.reg(fit_dyslex_full)
# 
# # using SIC in step 1 and AIC in step 2
# auto.beta.reg(fit_dyslex_full,crit1="SIC",crit2="AIC")
# 
# 
# ############################################
# ## 2) food expenditure data
# ## Application presented in Bayer and Cribari-Neto (2014b)
# data<-read.table("food.txt",h=T) 
# #attach(data)
# 
# set.seed(10)
# 
# # fitting the full model
# fit_food_full <- gamlss(Y~x2+x3+x4+x5+x6,family=BE)
# 
# # using the auto.beta.reg function with BQCV in step 1 and 632QCV in step 2
# auto.beta.reg(fit_food_full,crit1="BQCV",crit2="632QCV") 
# 
# # using the auto.beta.reg function with 632QCV and BQCV (just 50 resamples)
# auto.beta.reg(fit_food_full,crit1="632QCV",crit2="BQCV",B=50) 
# 
# ##########################################################################################

library(gamlss)
auto.beta.reg<-function(fitfull,
                        crit1="AIC",
                        crit2="R2_LRw",
                        B=200, # number of resamples for BQCV and 632QCV
                        alpha1= 0.4, # R2LEw
                        delta= 1, # R2LEw
                        alpha2= 0.6 # R2D
)
{
  print(" ",quote=F)
  print("Please wait!",quote=F)
  
  # link function for mu
  mulink<-function(x)
  { 
    link1<-make.link(fitfull$mu.link)
    return(link1$linkfun(y)) 
  }
  
  # log-likelihood function
  loglik<-function(mu,phi,y)
  {  
    return(sum(lgamma(phi)-lgamma(mu*phi)-lgamma((1-mu)*phi)+(mu*phi-1)*log(y)+((1-mu)*phi-1)*log(1-y)))
  }
  
  y<-fitfull$y
  M1<-model.matrix(fitfull)
  M<-M1[,2:ncol(M1)] # model matrix
  r<-ncol(M) 
  n<-fitfull$N
  
  # fitting null model
  fit_null = try(gamlss(y~+1, family=BE, trace=F, method=RS(),silent =T), silent = T) 
  loglik_null<- logLik(fit_null)
  lik_null<-exp(loglik_null)
  
  my<- mean(y) 
  link_y<- mulink(y) 
  sqt1<- sum((y-my)^2)
  
  loglik.max<- logLik(fit_null)
  
  k1<-length(fit_null$mu.coefficients)
  k2<-length(fit_null$sigma.coefficients)
  p<- k1+k2
  
  sqr1<- sum((y-fit_null$mu.fv)^2) 
  
  y_star <-sqrt(((y-fit_null$mu.fv)^2)/(fit_null$mu.fv*(1-fit_null$mu.fv))) # sigma^2=var(y)/(mu(1-mu))
  sqt2<- sum((y_star-mean(y_star))^2) 
  sqr2<- sum((y_star-fit_null$sigma.fv)^2) 
  
  # qualisty measures
  R2_FC <- max.R2_FC <- (cor(mulink(y),fit_null$mu.lp))^2 # Ferrari and Cribari-Neto (2004)
  R2_McF <- max.R2_McF <- 1-(loglik_null/loglik.max)  # McFadden (1974)
  R2_LR <- max.R2_LR <- 1-(lik_null/exp(loglik.max))^(2/n) # Long (1974) pag 104
  
  if(lik_null == Inf) 
  {  max.R2_LR <- 1       
     print("Problem! Linkelihood=Inf",quote=F)
  }
  
  R2_HS <- 1 - (sum((y-fit_null$mu.fv)^2)/sum((y-mean(y))^2)) # Hu and Shao (2008)
  
  # Model selection criteria
  max.R2_FC_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_FC)
  max.R2_McF_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_McF)
  max.R2_LR_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_LR)
  
  max.R2_HS0 <- 1 - ((n-1)/(n-(p)))*(sqr1/sqt1) # Hu and Shao (2008) (lambda_n=1)
  max.R2_HS1 <- 1 - ((n-1)/(n-log(n)*(p)))*(sqr1/sqt1) # Hu and Shao (2008)
  max.R2_HS2 <- 1 - ((n-1)/(n-sqrt(n)*(p)))*(sqr1/sqt1) # Hu and Shao (2008)
  
  min.AIC = -2*loglik.max + 2*(p) 
  min.AICc = -2*loglik.max + 2*p*(n/(n-p-1))
  min.SIC = -2*loglik.max + log(n)*(p)
  min.SICc = -2*loglik.max + (n*log(n)*p)/(n-p-1)
  min.HQ = -2*loglik.max + 2*p*log(log(n))
  min.HQc = -2*loglik.max+ 2*p*n*log(log(n))/(n-p-1)
  
  R2_RAYDm1 <- 1 - ((n-1)/(n-log(n)*(k1)))*(sqr1/sqt1)  
  R2_RAYDs0 <- 1 - ((n-1)/(n-(k2)))*(sqr2/sqt2) 
  max.R2_D <- (1-alpha2)*R2_RAYDm1 + alpha2*R2_RAYDs0 
  max.R2_LRw <- 1 - ((n-1)/(n-((1+alpha1)*k1+(1-alpha1)*k2)))^delta*(1-R2_LR)
  
  
  #####################################################################
  # Bootstrap-based criteria (Bayer and Cribari-Neto, 2014)
  if(crit1=="BQCV" || crit1=="632QCV" || crit2=="BQCV" || crit2=="632QCV" )
  {
    mu1<-fit_null$mu.fv
    sigma1<-fit_null$sigma.fv
    phi1<-(1-sigma1^2)/(sigma1^2)
    
    p1 <- mu1*phi1
    q1 <- (1-mu1)*phi1
    
    vloglik_star<-rep(0,B)
    k<-0
    contC<-0
    while(k<B)
    {
      y_star <- rbeta(n, p1, q1)
      fit_star<- try(gamlss(y_star~+1, 
                            family=BE, trace=F, method=RS(),silent = T), silent = T) 
      
      if(class(fit_star) != "try-error")
      {
        if ((fit_star$conv==FALSE)){
          contC=contC+1
          print("No convergence",quote=F)
        }else{ 
          
          mu_star<-fit_star$mu.fv
          sigma_star<-fit_star$sigma.fv
          phi_star<-(1-sigma_star^2)/(sigma_star^2)
          
          k<-k+1
          vloglik_star[k] <- -2*loglik(mu_star,phi_star,y)
        }
      }
    }
    min.BQCV <- mean(vloglik_star)
    min.BQCV2 <- 0.368*(-2*loglik(mu1,phi1,y))+0.632*min.BQCV
  }else{min.BQCV<-min.BQCV2<-BQCV<-BQCV2<-min.AIC}
  # Final of bootstrap-based criteria
  #####################################################################
  
  for(j in 1:r) 
  {
    Mr<-combn(r,j)
    
    for(i in 1:ncol(Mr))
    {
      regressors<-Mr[,i]
      #print(regressors)
      fit = try(gamlss(y~M[,regressors], 
                       family=BE, trace=F, method=RS(),silent = T), silent = T) 
      loglik.max<- logLik(fit)
      
      k1<-length(fit$mu.coefficients)
      k2<-length(fit$sigma.coefficients)
      p<- k1+k2
      
      sqr1<- sum((y-fit$mu.fv)^2) 
      
      y_star <-sqrt(((y-fit$mu.fv)^2)/(fit$mu.fv*(1-fit$mu.fv))) # sigma^2=var(y)/(mu(1-mu))
      sqt2<- sum((y_star-mean(y_star))^2) 
      sqr2<- sum((y_star-fit$sigma.fv)^2) 
      
      # Quality measures
      R2_FC <- (cor(mulink(y),fit$mu.lp))^2  # Ferrai and Cribari-Neto (2004)
      R2_McF <- 1-(loglik_null/loglik.max)  # McFadden (1974)
      R2_LR <- 1-(lik_null/exp(loglik.max))^(2/n) # Long (1974) pag 104 
      if(lik_null == Inf) 
      {  R2_LR <- 1       
         print("Problem! Linkelihood=Inf")
      }
      
      R2_HS <- 1 - (sum((y-fit$mu.fv)^2)/sum((y-mean(y))^2)) # Hu and Shao (2008)
      
      # Model selection criteria
      R2_FC_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_FC)
      R2_McF_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_McF)
      R2_LR_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_LR)
      
      R2_HS0 <- 1 - ((n-1)/(n-(p)))*(sqr1/sqt1) # Hu and Shao (2008) (lambda_n=1)
      R2_HS1 <- 1 - ((n-1)/(n-log(n)*(p)))*(sqr1/sqt1) # Hu and Shao (2008)
      R2_HS2 <- 1 - ((n-1)/(n-sqrt(n)*(p)))*(sqr1/sqt1) # Hu and Shao (2008)
      
      eAIC = -2*loglik.max + 2*(p)
      eAICc = -2*loglik.max + 2*p*(n/(n-p-1))
      eSIC = -2*loglik.max + log(n)*(p)
      eSICc = -2*loglik.max + (n*log(n)*p)/(n-p-1)
      eHQ = -2*loglik.max + 2*p*log(log(n))
      eHQc = -2*loglik.max+ 2*p*n*log(log(n))/(n-p-1)
      
      R2_RAYDm1 <- 1 - ((n-1)/(n-log(n)*(k1)))*(sqr1/sqt1) 
      R2_RAYDs0 <- 1 - ((n-1)/(n-(k2)))*(sqr2/sqt2) 
      R2_D <- (1-alpha2)*R2_RAYDm1 + alpha2*R2_RAYDs0 # Bayer and Cribari-Neto (2014a)
      
      R2_LRw <- 1 - ((n-1)/(n-((1+alpha1)*k1+(1-alpha1)*k2)))^delta*(1-R2_LR) # Bayer and Cribari-Neto (2014a)
      
      
      #####################################################################
      # Bootstrap-based criteria (Bayer and Cribari-Neto, 2014b)
      if(crit1=="BQCV" || crit1=="632QCV" )
      {
        mu1<-fit$mu.fv
        sigma1<-fit$sigma.fv
        phi1<-(1-sigma1^2)/(sigma1^2)
        
        p1 <- mu1*phi1
        q1 <- (1-mu1)*phi1
        
        vloglik_star<-rep(0,B)
        k<-0
        contC<-0
        while(k<B)
        {
          y_star <- rbeta(n, p1, q1)
          fit_star<- try(gamlss(y_star~M[,regressors], 
                                family=BE, trace=F, method=RS(),silent = T), silent = T) 
          
          if(class(fit_star) != "try-error")
          {
            if ((fit_star$conv==FALSE)){
              contC=contC+1
              print("No convergence",quote=F)
            }else{ 
              
              mu_star<-fit_star$mu.fv
              sigma_star<-fit_star$sigma.fv
              phi_star<-(1-sigma_star^2)/(sigma_star^2)
              
              k<-k+1
              vloglik_star[k] <- -2*loglik(mu_star,phi_star,y)
            }
          }
        }
        BQCV <- mean(vloglik_star)
        BQCV2 <- 0.368*(-2*loglik(mu1,phi1,y))+0.632*BQCV
      }else{BQCV<-BQCV2<-eAIC}
      # Final of bootstrap-based criteria
      #####################################################################
      
      if(R2_FC_aj>max.R2_FC_aj)
      {  max.R2_FC_aj<-R2_FC_aj
         mod.max.R2_FC_aj<-regressors
         mod.R2_FC_aj<-fit
      }
      if(R2_McF_aj>max.R2_McF_aj)
      {  max.R2_McF_aj<-R2_McF_aj
         mod.max.R2_McF_aj<-regressors
         mod.R2_McF_aj<-fit
      }
      if(R2_LR_aj>max.R2_LR_aj)
      {  max.R2_LR_aj<-R2_LR_aj
         mod.max.R2_LR_aj<-regressors
         mod.R2_LR_aj<-fit
      }
      if(R2_HS0>max.R2_HS0)
      {  max.R2_HS0<-R2_HS0
         mod.max.R2_HS0<-regressors
         mod.R2_HS0<-fit
      }
      if(R2_HS1>max.R2_HS1)
      {  max.R2_HS1<-R2_HS1
         mod.max.R2_HS1<-regressors
         mod.R2_HS1<-fit
      }
      if(R2_HS2>max.R2_HS2)
      {  max.R2_HS2<-R2_HS2
         mod.max.R2_HS2<-regressors
         mod.R2_HS2<-fit
      }
      
      if(eAIC<min.AIC)
      {  min.AIC<-eAIC
         mod.min.AIC<-regressors
         mod.AIC<-fit
      }
      if(eAICc<min.AICc)
      {  min.AICc<-eAICc
         mod.min.AICc<-regressors
         mod.AICc<-fit
      }
      if(eSIC<min.SIC)
      {  min.SIC<-eSIC
         mod.min.SIC<-regressors
         mod.SIC<-fit
      }
      if(eSICc<min.SICc)
      {  min.SICc<-eSICc
         mod.min.SICc<-regressors
         mod.SICc<-fit
      }
      if(eHQ<min.HQ)
      {  min.HQ<-eHQ
         mod.min.HQ<-regressors
         mod.HQ<-fit
      }
      if(eHQc<min.HQc)
      {  min.HQc<-eHQc
         mod.min.HQc<-regressors
         mod.HQc<-fit
      }
      
      if(R2_D > max.R2_D)
      {  max.R2_D<-R2_D
         mod.max.R2_D<-regressors
         mod.R2_D<-fit
      }
      if(R2_LRw > max.R2_LRw)
      {  max.R2_LRw <-R2_LRw
         mod.max.R2_LRw <-regressors
         mod.R2_LRw <-fit
      }
      if(BQCV<min.BQCV)
      {  min.BQCV<-BQCV
         mod.min.BQCV<-regressors
         mod.BQCV<-fit
      }
      if(BQCV2<min.BQCV2)
      {  min.BQCV2<-BQCV2
         mod.min.BQCV2<-regressors
         mod.BQCV2<-fit
      }
    } # end for i mean
  } # end for j mean
  
  if(crit1 == "R2_FC_aj")
  {  
    model.mean<-mod.max.R2_FC_aj
  }
  if(crit1 == "R2_McF_aj")
  {  
    model.mean<-mod.max.R2_McF_aj
  }
  if(crit1 == "R2_LR_aj")
  {  
    model.mean<-mod.max.R2_LR_aj
  }
  if(crit1 == "R2_HS0")
  {  
    model.mean<-mod.max.R2_HS0
  }
  if(crit1 == "R2_HS1")
  {  
    model.mean<-mod.max.R2_HS1
  }
  if(crit1 == "R2_HS2")
  {  
    model.mean<-mod.max.R2_HS2
  }
  
  if(crit1 == "AIC")
  {  
    model.mean<-mod.min.AIC
  }
  if(crit1 == "AICc")
  {  
    model.mean<-mod.min.AICc
  }    
  if(crit1 == "SIC")
  { 
    model.mean<-mod.min.SIC
  }
  if(crit1 == "SICc")
  {  
    model.mean<-mod.min.SICc
  }
  if(crit1 == "HQ")
  {  
    model.mean<-mod.min.HQ
  }
  if(crit1 == "HQc")
  {  
    model.mean<-mod.min.HQc
  }
  
  if(crit1 == "R2_D" )
  {  
    model.mean<-mod.max.R2_D
  }
  if(crit1 == "R2_LRw" )
  {  
    model.mean<-mod.max.R2_LRw
  }
  if(crit1 == "BQCV")
  {  
    model.mean<-mod.min.BQCV
  }
  if(crit1 == "632QCV")
  {  
    model.mean<-mod.min.BQCV2
  }
  
  print("###########################")
  print("Mean submodel")
  print(model.mean)
  
  # dispersion submodel
  for(j in 1:r) 
  {
    Mr<-combn(r,j)
    
    for(i in 1:ncol(Mr))
    {
      regressors<-Mr[,i]
      #print(regressors)
      fit = try(gamlss(y~M[,model.mean], 
                       sigma.formula=~M[,regressors],
                       family=BE, trace=F, method=RS(),silent = T), silent = T) 
      loglik.max<- logLik(fit)
      
      k1<-length(fit$mu.coefficients)
      k2<-length(fit$sigma.coefficients)
      p<- k1+k2
      
      sqr1<- sum((y-fit$mu.fv)^2) 
      
      y_star <-sqrt(((y-fit$mu.fv)^2)/(fit$mu.fv*(1-fit$mu.fv))) # sigma^2=var(y)/(mu(1-mu))
      sqt2<- sum((y_star-mean(y_star))^2) 
      sqr2<- sum((y_star-fit$sigma.fv)^2) 
      
      # Quality measures
      R2_FC <- (cor(mulink(y),fit$mu.lp))^2  # Ferrari and Criabri-Neto (2004)
      R2_McF <- 1-(loglik_null/loglik.max)  # McFadden (1974)
      R2_LR <- 1-(lik_null/exp(loglik.max))^(2/n) # Long (1974) pag 104
      if(lik_null == Inf) 
      {  R2_LR <- 1     
         print("Problem! Linkelihood=Inf")
      }
      
      R2_HS <- 1 - (sum((y-fit$mu.fv)^2)/sum((y-mean(y))^2)) # Hu and Shao (2008)
      
      # Model selection criteria
      R2_FC_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_FC)
      R2_McF_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_McF)
      R2_LR_aj <- 1 - ((n-1)/(n-(p)))* (1-R2_LR)
      
      R2_HS0 <- 1 - ((n-1)/(n-(p)))*(sqr1/sqt1) # Hu and Shao (2008) (lambda_n=1)
      R2_HS1 <- 1 - ((n-1)/(n-log(n)*(p)))*(sqr1/sqt1) # Hu and Shao (2008)
      R2_HS2 <- 1 - ((n-1)/(n-sqrt(n)*(p)))*(sqr1/sqt1) # Hu and Shao (2008)
      
      eAIC = -2*loglik.max + 2*(p) 
      eAICc = -2*loglik.max + 2*p*(n/(n-p-1))
      eSIC = -2*loglik.max + log(n)*(p)
      eSICc = -2*loglik.max + (n*log(n)*p)/(n-p-1)
      eHQ = -2*loglik.max + 2*p*log(log(n))
      eHQc = -2*loglik.max+ 2*p*n*log(log(n))/(n-p-1)
      
      R2_RAYDm1 <- 1 - ((n-1)/(n-log(n)*(k1)))*(sqr1/sqt1) 
      R2_RAYDs0 <- 1 - ((n-1)/(n-(k2)))*(sqr2/sqt2) 
      R2_D <- (1-alpha2)*R2_RAYDm1 + alpha2*R2_RAYDs0 # Bayer and Cribari-Neto (2014a)
      
      R2_LRw <- 1 - ((n-1)/(n-((1+alpha1)*k1+(1-alpha1)*k2)))^delta*(1-R2_LR) # Bayer and Cribari-Neto (2014a)
      
      #####################################################################
      # Bootstrap-based criteria (Bayer and Cribari-Neto, 2014b)
      if(crit2=="BQCV" || crit2=="632QCV" )
      {
        mu1<-fit$mu.fv
        sigma1<-fit$sigma.fv
        phi1<-(1-sigma1^2)/(sigma1^2)
        
        p1 <- mu1*phi1
        q1 <- (1-mu1)*phi1
        
        vloglik_star<-rep(0,B)
        k<-0
        contC<-0
        while(k<B)
        {
          y_star <- rbeta(n, p1, q1)
          fit_star<- try(gamlss(y_star~M[,model.mean], 
                                sigma.formula=~M[,regressors],
                                family=BE, trace=F, method=RS(),silent = T), silent = T) 
          
          if(class(fit_star) != "try-error")
          {
            if ((fit_star$conv==FALSE)){
              contC=contC+1
              print("No convergence",quote=F)
            }else{ 
              
              mu_star<-fit_star$mu.fv
              sigma_star<-fit_star$sigma.fv
              phi_star<-(1-sigma_star^2)/(sigma_star^2)
              
              k<-k+1
              vloglik_star[k] <- -2*loglik(mu_star,phi_star,y)
            }
          }
        }
        BQCV <- mean(vloglik_star)
        BQCV2 <- 0.368*(-2*loglik(mu1,phi1,y))+0.632*BQCV
      }else{BQCV<-BQCV2<-eAIC}
      # Final of bootstrap-based criteria
      #####################################################################
      
      if(R2_FC_aj>max.R2_FC_aj)
      {  max.R2_FC_aj<-R2_FC_aj
         mod.max.R2_FC_aj<-regressors
         mod.R2_FC_aj<-fit
      }
      if(R2_McF_aj>max.R2_McF_aj)
      {  max.R2_McF_aj<-R2_McF_aj
         mod.max.R2_McF_aj<-regressors
         mod.R2_McF_aj<-fit
      }
      if(R2_LR_aj>max.R2_LR_aj)
      {  max.R2_LR_aj<-R2_LR_aj
         mod.max.R2_LR_aj<-regressors
         mod.R2_LR_aj<-fit
      }
      if(R2_HS0>max.R2_HS0)
      {  max.R2_HS0<-R2_HS0
         mod.max.R2_HS0<-regressors
         mod.R2_HS0<-fit
      }
      if(R2_HS1>max.R2_HS1)
      {  max.R2_HS1<-R2_HS1
         mod.max.R2_HS1<-regressors
         mod.R2_HS1<-fit
      }
      if(R2_HS2>max.R2_HS2)
      {  max.R2_HS2<-R2_HS2
         mod.max.R2_HS2<-regressors
         mod.R2_HS2<-fit
      }
      
      if(eAIC<min.AIC)
      {  min.AIC<-eAIC
         mod.min.AIC<-regressors
         mod.AIC<-fit
      }
      if(eAICc<min.AICc)
      {  min.AICc<-eAICc
         mod.min.AICc<-regressors
         mod.AICc<-fit
      }     
      if(eSIC<min.SIC)
      {  min.SIC<-eSIC
         mod.min.SIC<-regressors
         mod.SIC<-fit
      }
      if(eSICc<min.SICc)
      {  min.SICc<-eSICc
         mod.min.SICc<-regressors
         mod.SICc<-fit
      }
      if(eHQ<min.HQ)
      {  min.HQ<-eHQ
         mod.min.HQ<-regressors
         mod.HQ<-fit
      }
      if(eHQc<min.HQc)
      {  min.HQc<-eHQc
         mod.min.HQc<-regressors
         mod.HQc<-fit
      }
      
      if(R2_D > max.R2_D)
      {  max.R2_D<-R2_D
         mod.max.R2_D<-regressors
         mod.R2_D<-fit
      }
      if(R2_LRw > max.R2_LRw)
      {  max.R2_LRw <-R2_LRw
         mod.max.R2_LRw <-regressors
         mod.R2_LRw <-fit
      }
      if(BQCV<min.BQCV)
      {  min.BQCV<-BQCV
         mod.min.BQCV<-regressors
         mod.BQCV<-fit
      }
      if(BQCV2<min.BQCV2)
      {  min.BQCV2<-BQCV2
         mod.min.BQCV2<-regressors
         mod.BQCV2<-fit
      }
      
    } # end for i mean
  } # end for j mean
  
  
  if(crit2 == "R2_FC_aj")
  {  
    model.sigma<-mod.max.R2_FC_aj
  }
  if(crit2 == "R2_McF_aj")
  {  
    model.sigma<-mod.max.R2_McF_aj
  }
  if(crit2 == "R2_LR_aj")
  {  
    model.sigma<-mod.max.R2_LR_aj
  }
  if(crit2 == "R2_HS0")
  {  
    model.sigma<-mod.max.R2_HS0
  }
  if(crit2 == "R2_HS1")
  {  
    model.sigma<-mod.max.R2_HS1
  }
  if(crit2 == "R2_HS2")
  {  
    model.sigma<-mod.max.R2_HS2
  }
  
  if(crit2 == "AIC")
  {  
    model.sigma<-mod.min.AIC
  }
  if(crit2 == "AICc")
  {  
    model.sigma<-mod.min.AICc
  }      
  if(crit2 == "SIC")
  { 
    model.sigma<-mod.min.SIC
  }
  if(crit2 == "SICc")
  {  
    model.sigma<-mod.min.SICc
  }
  if(crit2 == "HQ")
  {  
    model.sigma<-mod.min.HQ
  }
  if(crit2 == "HQc")
  {  
    model.sigma<-mod.min.HQc
  }
  
  if(crit2 == "R2_D" )
  {  
    model.sigma<-mod.max.R2_D
  }
  if(crit2 == "R2_LRw" )
  {  
    model.sigma<-mod.max.R2_LRw
  }
  if(crit2 == "BQCV")
  {  
    model.sigma<-mod.min.BQCV
  }
  if(crit2 == "632QCV")
  {  
    model.sigma<-mod.min.BQCV2
  }
  print("###########################")
  print("Dispersion submodel")
  print(model.sigma)
  
  fit.final = gamlss(y~M[,model.mean], 
                     sigma.formula=~M[,model.sigma],
                     family=BE, trace=F, method=RS())
  
  print(summary(fit.final))
  print("Quality measures",quote=F)
  print(c("R2_FC=",R2_FC),quote=F)
  print(c("R2_LR=",R2_LR),quote=F)
  print(c("Using:", crit1,crit2),quote=F)
  print(" ",quote=F)
  
  return(fit.final)
  
} # end function



